import { AlertCircle, Loader2, Users } from "lucide-react";
import { useQuery } from "@tanstack/react-query";
import { useState } from "react";
import { apiRequest } from "@/lib/queryClient";
import { EmptyState } from "../EmptyState";
import { StateDisplay } from "../StateDisplay";


interface Campaign {
  id: string;
  channelId: string | null;
  createdBy: string | null;
  name: string;
  description: string | null;
  campaignType: string;
  type: string;
  apiType: string | null;
  templateId: string | null;
  templateName: string | null;
  templateLanguage: string | null;
  variableMapping: Record<string, any>;
  contactGroups: string[];
  csvData: any[];
  apiKey: string | null;
  apiEndpoint: string | null;
  status: string | null;
  scheduledAt: string | null;
  recipientCount: number;
  sentCount: number;
  deliveredCount: number;
  readCount: number;
  repliedCount: number;
  failedCount: number;
  completedAt: string | null;
  createdAt: string;
  updatedAt: string;
}

interface CampaignsProps {
  userId: string;
}

interface CampaignsResponse {
  data: Campaign[];
  total: number;
  page: number;
  limit: number;
}

export default function Campaigns({ userId }: CampaignsProps) {
  const [page, setPage] = useState<number>(1);
  const [limit, setLimit] = useState(10);

  const { data, isLoading, isError, error } = useQuery<
    CampaignsResponse,
    Error
  >({
    queryKey: ["campaigns", userId, page, limit],
    queryFn: async () => {
      const res = await apiRequest("POST", "/api/getCampaignsByUserId", {
        userId,
        page,
        limit,
      });
      const json: CampaignsResponse = await res.json();
      return json;
    },
    enabled: !!userId,
    keepPreviousData: true,
  });

  const campaigns = data?.data ?? [];
  const total = data?.total ?? 0;
  const totalPages = Math.ceil(total / limit);

  if (isLoading)
    return (
      <div className="flex items-center justify-center py-10 text-muted-foreground">
        <Loader2 className="w-5 h-5 mr-2 animate-spin" /> Loading campaigns...
      </div>
    );

  // Error State
  if (isError) {
    return (
      <StateDisplay
        variant="error"
        icon={AlertCircle}
        title="Failed to Load Channels"
        description={"Something went wrong while fetching Channels."}
        buttonText="Try Again"
        onButtonClick={() => window.location.reload()}
      />
    );
  }

  // Empty State
  if (campaigns.length === 0) {
    return (
      <StateDisplay
        icon={Users}
        title="No Channels Yet"
        description="Start building your team by inviting members. They'll appear here once added."
        buttonText="Invite Team Member"
      />
    );
  }

  return (
    <div>
      <div className="overflow-x-auto">
        <table className="min-w-full border border-gray-200 bg-white rounded-lg shadow-sm">
          <thead className="bg-gray-100 text-left text-sm font-semibold text-gray-700">
            <tr>
              <th className="py-3 px-4 border-b">Name</th>
              <th className="py-3 px-4 border-b">Type</th>
              <th className="py-3 px-4 border-b">Status</th>
              <th className="py-3 px-4 border-b">Scheduled At</th>
              <th className="py-3 px-4 border-b">Created At</th>
            </tr>
          </thead>
          <tbody>
            {campaigns.map((campaign) => (
              <tr
                key={campaign.id}
                className="hover:bg-gray-50 transition-colors text-sm text-gray-700"
              >
                <td className="py-3 px-4 border-b">{campaign.name}</td>
                <td className="py-3 px-4 border-b">{campaign.type}</td>
                <td className="py-3 px-4 border-b">
                  {campaign.status === "scheduled" ? (
                    <span className="px-2 py-1 rounded text-xs bg-blue-100 text-blue-700">
                      {campaign.status}
                    </span>
                  ) : campaign.status === "completed" ? (
                    <span className="px-2 py-1 rounded text-xs bg-green-100 text-green-700">
                      {campaign.status}
                    </span>
                  ) : (
                    <span className="px-2 py-1 rounded text-xs bg-yellow-100 text-yellow-700">
                      {campaign.status || "unknown"}
                    </span>
                  )}
                </td>
                <td className="py-3 px-4 border-b">
                  {campaign.scheduledAt
                    ? new Date(campaign.scheduledAt).toLocaleString()
                    : "-"}
                </td>
                <td className="py-3 px-4 border-b">
                  {new Date(campaign.createdAt).toLocaleString()}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {/* Pagination (Fully Responsive) */}
      <div className="w-full mt-6 flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        {/* LEFT SIDE → Showing Results + Per Page */}
        <div className="flex flex-col sm:flex-row sm:items-center gap-2">
          <span className="text-sm text-gray-700">
            Showing {(page - 1) * limit + 1} to {Math.min(page * limit, total)}{" "}
            of {total} campaigns
          </span>

          {/* Per Page Dropdown */}
          <select
            value={limit}
            onChange={(e) => {
              setLimit(Number(e.target.value));
              setPage(1);
            }}
            className="border px-3 py-2 rounded-md text-sm w-24"
          >
            <option value={5}>5</option>
            <option value={10}>10</option>
            <option value={20}>20</option>
            <option value={50}>50</option>
          </select>
        </div>

        {/* RIGHT SIDE → Pagination Buttons */}
        <div className="flex items-center justify-center sm:justify-end gap-2">
          <button
            className="px-3 py-1 border rounded disabled:opacity-50"
            onClick={() => setPage((p) => Math.max(p - 1, 1))}
            disabled={page === 1}
          >
            Previous
          </button>

          <span className="bg-green-600 text-white px-4 py-2 rounded-md text-sm font-medium">
            {page}
          </span>

          <button
            className="px-3 py-1 border rounded disabled:opacity-50"
            onClick={() => setPage((p) => Math.min(p + 1, totalPages))}
            disabled={page === totalPages}
          >
            Next
          </button>
        </div>
      </div>
    </div>
  );
}
